/**
 * \file CometIndexApplication.cpp
 * \brief Runs comet -i
 *****************************************************************************/
#include "CometIndexApplication.h"
#include "io/DelimitedFile.h"
#include "io/DelimitedFileWriter.h"
#include "util/AminoAcidUtil.h"
#include "util/CarpStreamBuf.h"
#include "util/FileUtils.h"
#include "util/Params.h"
#include "util/StringUtils.h"

using namespace std;

/**
 * \returns a blank CometApplication object
 */
CometIndexApplication::CometIndexApplication() {
}

/**
 * Destructor
 */
CometIndexApplication::~CometIndexApplication() {
}

/**
 * main method for CometIndexApplication
 */
int CometIndexApplication::main(int argc, char **argv) {
    return main();
}

int CometIndexApplication::main() {

    /* Re-route stderr to log file */
    CarpStreamBuf buffer;
    streambuf *old = std::cerr.rdbuf();
    std::cerr.rdbuf(&buffer);

    /* set parameters */
    vector<string> input_files;
    vector<InputFileInfo *> pv_input_files;
    setCometParameters(input_files, pv_input_files);
    searchManager_.AddInputFiles(pv_input_files);
    // Create index 
    bool success = false;
    // Update the Comet parameters for which type of 
    // index should be created. Index is either ion fragment
    // or peptide.But both have to be set,
    // The create_fragment_index is a hidden parameter for crux.
    if (Params::GetBool("create_peptide_index")) {
      searchManager_.SetParam("create_peptide_index", "1", 1);
      searchManager_.SetParam("create_fragment_index", "0", 0);
      success = searchManager_.CreatePeptideIndex();
    } else {
      searchManager_.SetParam("create_peptide_index", "0", 0);
      searchManager_.SetParam("create_fragment_index", "1", 1);
      success = searchManager_.CreateFragmentIndex();
    }

    /* Recover stderr */
    std::cerr.rdbuf(old);

    return success ? 0 : 1;
}


/**
 * Sets the parameters for the Comet application using the crux parameters
 */
void CometIndexApplication::setCometParameters(
  const vector<string>& spec_files,
  vector<InputFileInfo*>& pvInputFiles ///<vector of input spectra files
) {
  CometApplication::setCometParameters(spec_files, pvInputFiles);
  // Update the Comet parameters for which type of 
  // index should be created. Index is either ion fragment
  // or peptide, but both heave to be set.
  // The create_fragment_index is a hidden parameter for crux.
  if (Params::GetBool("create_fragment_index")) {
    searchManager_.SetParam("create_peptide_index", "0", 0);
    searchManager_.SetParam("create_fragment_index", "1", 0);
  }
  if (Params::GetBool("create_peptide_index")) {
    searchManager_.SetParam("create_peptide_index", "1", 0);
    searchManager_.SetParam("create_fragment_index", "0", 0);
  }
}


/**
 * \returns the command name for CometApplication
 */
string CometIndexApplication::getName() const { return "comet-index"; }

/**
 * \returns the description for CometIndexApplication
 */
string CometIndexApplication::getDescription() const {
    return "[[nohtml:Create an index from a sequence database.]]"
           "[[html:<p>This command creates an index from a sequence database. "
           "By default, the index is of fragment ions. An index of peptides can be "
           "generated by setting the <code>--create_peptide_index T</code> option. "
           "This indexing engine was developed by Jimmy Eng at the University of Washington "
           "Proteomics Resource.</p>]]";
}

/**
 * \returns the command options
 */
vector<string> CometIndexApplication::getOptions() const {

  vector<string> options = CometApplication::getOptions();
  // Fragment ion fragment indexing
  options.insert(options.begin(), "create_fragment_index");
  // Peptide indexing
  options.insert(options.begin(), "create_peptide_index");
  return options;

}

/**
 * \returns the command arguments
 */
vector<string> CometIndexApplication::getArgs() const {
    string arr[] = {"index_database_name"};
    return vector<string>(arr, arr + sizeof(arr) / sizeof(string));
}

/**
 * \returns the command options
 */

/**
 * \returns the command outputs
 */
vector<pair<string, string>> CometIndexApplication::getOutputs() const {
    vector<pair<string, string>> outputs;
    outputs.push_back(make_pair("comet.params.txt",
                                "a file containing the name and value of all "
                                "parameters/options for the "
                                "current operation. Not all parameters in the "
                                "file may have been used in "
                                "the operation. The resulting file can be used "
                                "with the --parameter-file "
                                "option for other crux programs."));
    outputs.push_back(make_pair(
        "comet.log.txt",
        "a log file containing a copy of all messages that were printed to "
        "standard error."));
    return outputs;
}

COMMAND_T CometIndexApplication::getCommand() const {
    return COMET_INDEX_COMMAND;
}

void CometIndexApplication::processParams() {

  // We name the database file parameter differerntly so
  // we can have a different usage message.
  // We have to switch to the name Comet knows it by.
  Params::Set("database_name", Params::GetString("index_database_name"));
}

/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 2
 * End:
 */
